/*
 * MACRO to find polarized distribution of fluorophore tagged protein
 * the macro works by finding the cell ROIs using Analyze>>Analyze Particles. 
 * measures Centroid (center of ROI according to X,Y pixel distribution) and Center of Mass (intensity weighted center of ROI)
 * calculates Polarization_Index by finding the distance between Centroid and Center of Mass
 * calculates Polarization_Angle by finding the angle of the line connecting the Centroid and Center of Mass w.r.t. horizontal
 * make a folder to save cell ROIs for corresponding image stack
 * Open the FLUORESCENCE IMAGES of cells in Fiji(ImageJ)
 * Rename one of the channels as "A" and the other channel as "B" (Assign "A" to the channel that shows the whole cell clearly)
 * if images do not have the pixel-to-micron scale set, using Set Scale, convert distance in pixels to microns in the image (and apply to Global if useful)
 * (This ensures that images from any microscope can use this macro to get comparable data)
 * Get an idea of the minimum and maximum cell spread area (in microns^2) for user input
 * Alternatively, if images are present as separate TIFF files in a folder, browse there and follow rest of the instructions
 * Run the Macro, have a beer
 * Copy results from Log file, paste in an Spreadsheet, close Log file
 * MACRO ONLY WORKS FOR IMAGE STACKS (timelapse movies)
*/
thresholdingMethod="Default";
gammaFactor=2.0;

run("Options...", "iterations=1 count=1 do=Nothing");
ROIsaveDestination = getDirectory("Browse to Folder where you want to save the ROIs");

nROI=roiManager("count"); //find number of ROIs stored
if (nROI>0) { roiManager("deselect"); roiManager("delete"); } //clear previously stored ROIs

//CONFIRM CELL AREA RANGE FROM USER
cellAreaRangeMin=40;  //in um^2
cellAreaRangeMax=400; //in um^2
Dialog.create("User Input");
Dialog.addNumber("Minimum Area of Cells to consider (in microns):", cellAreaRangeMin);
Dialog.addNumber("Maximum Area of Cells to consider (in microns):", cellAreaRangeMax);
Dialog.addCheckbox("Show masks", false);
Dialog.addCheckbox("Show Displacement vectors", false);
Dialog.addCheckbox("Keep the cropped cell image stacks after running macro", false);
Dialog.addCheckbox("To enhance contrast for better polarization values, use Gamma>1?", false);
Dialog.show();
chkbx1=Dialog.getCheckbox(); //to display mask images, i.e., not close them
chkbx2=Dialog.getCheckbox(); //to display the image containing the displacement vectors
chkbx3=Dialog.getCheckbox(); //to display the user selected ROIs containing individual cells
chkbx4=Dialog.getCheckbox(); //to selectively increase higher signals and decrease lower signals

if (chkbx4==false) { gammaFactor=1.0; }
if (chkbx4==true) { Dialog.addNumber("Use Gamma value:", gammaFactor); Dialog.show(); }

nOpenImgs=nImages;
if (nOpenImgs==0)
{
	//OPEN RELEVANT IMAGE SEQUENCES
	dirA = getDirectory("Browse to Folder where Channel A images are kept");
	run("Image Sequence...", "select=["+dirA+"] dir=["+dirA+"] sort");
	rename("A");
	dirB = getDirectory("Browse to Folder where corresponding Channel B images are kept");
	run("Image Sequence...", "select=["+dirB+"] dir=["+dirB+"] sort");
	rename("B");
}

//GET THE BACKGROUND INTENSITY
selectWindow("A");
run("Set Measurements...", "mean standard redirect=None decimal=3");
backg=2.03; sd=0.448; run("Clear Results"); setTool("rectangle");
waitForUser("In the image 'A', make an ROI of any size in a \ncell-free space (to get background intensity),\nAdd to ROI Manager, then hit OK");
if (roiManager("count")==0) { waitForUser("You didn't add to ROI_Manager, add and hit OK"); }
selectWindow("A"); roiManager("Select", 0); roiManager("Measure");
backg=parseFloat(getResultString("Mean", 0));
sd=parseFloat(getResultString("StdDev", 0));
roiManager("Select", 0); roiManager("Delete"); run("Clear Results");

//MERGE THE TWO FLUORESCENCE CHANNELS AND GET MAX INTENSITY PROJECTION TO GET EXTENT OF CELL MOVEMENT
run("Merge Channels...", "c1=B c2=A create keep"); imgMerged=getTitle();
run("Z Project...", "projection=[Max Intensity]");
selectWindow(imgMerged); close();
setTool("freehand");
waitForUser("Draw ROIs enclosing trajectories of relevant cells \n# Only 1 CELL per ROI. \n# CELL must have signal in both channels. \nAdd to ROI Manager, then hit OK");
if (roiManager("count")==0) { waitForUser("None added, please add cell trajectory ROIs to ROI Manager, then hit OK"); }
selectWindow("MAX_"+imgMerged); close();

//DUPLICATE ROIs MARKED BY USER, MAKE FAKE BACKGROUND WITH REAL NUMBERS OUTSIDE CELL ROIs
nCELLS=roiManager("count");
for (i=1; i<=nCELLS; i++)
{
	selectWindow("A"); roiManager("Select", (i-1)); run("Duplicate...", "title=A_"+i+" duplicate");
	selectWindow("A_"+i); run("Make Inverse"); run("Multiply...", "value=0 stack");
	run("Add...", "value="+backg+" stack"); run("Add Specified Noise...", "stack standard="+sd); run("Select None");
	
	selectWindow("B"); roiManager("Select", (i-1)); run("Duplicate...", "title=B_"+i+" duplicate");
	selectWindow("B_"+i); run("Make Inverse"); run("Multiply...", "value=0 stack");
	run("Add...", "value="+backg+" stack"); run("Add Specified Noise...", "stack standard="+sd); run("Select None");
	
	selectWindow("A_"+i); run("Select None"); run("Duplicate...", "title=mask-image_"+i+" duplicate"); //duplicate to make masks out of Channel A images to get cell shape
	
	//GENERATE CONTRAST ENHANCED DUPLICATES FOR POLARIZATION CALCULATION
	selectWindow("A_"+i); run("Select None"); run("Duplicate...", "title=AA_"+i+" duplicate");
	selectWindow("AA_"+i); run("Gamma...", "value="+gammaFactor+" stack");
	selectWindow("B_"+i); run("Select None"); run("Duplicate...", "title=BB_"+i+" duplicate");
	selectWindow("BB_"+i); run("Gamma...", "value="+gammaFactor+" stack");
}

//CLOSE MAIN IMAGES, KEEP ONLY DUPLICATES. SAVE ROIs
selectWindow("A"); //close();
selectWindow("B"); //close();
roiManager("save", ROIsaveDestination+"ROIset_AllCells.zip");
roiManager("Deselect"); roiManager("Delete");

run("Set Measurements...", "area centroid center fit shape integrated redirect=None decimal=3");

for (i=1; i<=nCELLS; i++)
{
	//CREATE BINARY MASKS FOR Analyze particles FUNCTION
	selectWindow("mask-image_"+i);  ns=nSlices; bd=bitDepth; getVoxelSize(widthPxl, heightPxl, depthPxl, unitPxl);
	run("Select None");
	run("Smooth", "stack"); //smooth images for easier thresholding
	setAutoThreshold(thresholdingMethod+" dark stack");
	run("Convert to Mask", "method="+thresholdingMethod+" background=Dark");
	run("Fill Holes", "stack");
	run("Analyze Particles...", "size="+cellAreaRangeMin+"-"+cellAreaRangeMax+" add stack");
	roiManager("save", ROIsaveDestination+"ROIset_Cell-"+i+"_Outline.zip");
	
	//CREATE ARRAYS TO STORE SHAPE DETAILS
	nROI=roiManager("count"); //find number of ROIs stored
	Area=newArray(nROI); Xc=newArray(nROI); Yc=newArray(nROI); Circularity=newArray(nROI);
	
	//DECLARE ARRAYS TO STORE CENTERS OF MASS FOR BOTH CHANNELS, ELLIPSE PARAMETERS
	XcMA=newArray(nROI); YcMA=newArray(nROI); XcMB=newArray(nROI); YcMB=newArray(nROI);
	Major=newArray(nROI); Angle=newArray(nROI);
	FintA=newArray(nROI); BintA=newArray(nROI); FintB=newArray(nROI); BintB=newArray(nROI); FrameN=newArray(nROI);
	
	//MAIN OPERATIONS: OBTAIN CENTERS OF MASS AND BISECTION
	
	//MEASURE AND STORE CELL SHAPE DETAILS IN THE ARRAYS
	run("Clear Results");
	for (j=0; j<nROI; j++)
	{
		selectWindow("AA_"+i); roiManager("Select", j); run("Measure");
		Area[j]=parseFloat(getResultString("Area", j));
		Xc[j]=parseFloat(getResultString("X", j));
		Yc[j]=parseFloat(getResultString("Y", j));
		XcMA[j]=parseFloat(getResultString("XM", j));
		YcMA[j]=parseFloat(getResultString("YM", j));
		Circularity[j]=parseFloat(getResultString("Circ.", j));
		Major[j]=parseFloat(getResultString("Major", j));
		Angle[j]=parseFloat(getResultString("Angle", j));
		Table.deleteRows(j, j);
		selectWindow("BB_"+i); roiManager("Select", j); run("Measure");
		XcMB[j]=parseFloat(getResultString("XM", j));
		YcMB[j]=parseFloat(getResultString("YM", j));
	}
	
	//FIT ELLIPSES ON THE CELL ROIs, AND BISECT CELL ROIs ALONG THE MINOR AXIS
	selectWindow("A_"+i); run("32-bit");
	selectWindow("B_"+i); run("32-bit");
	selectWindow("mask-image_"+i); run("Select None"); run("Multiply...", "value=0.000 stack");
	for (j=0; j<nROI; j++)
	{
		selectWindow("A_"+i); roiManager("Select", j); run("Make Inverse"); run("Multiply...", "value=0.000 slice"); run("Divide...", "value=0.000 slice"); //make channel A image background NaN
		selectWindow("B_"+i); roiManager("Select", j); run("Make Inverse"); run("Multiply...", "value=0.000 slice"); run("Divide...", "value=0.000 slice"); //make channel B image background NaN
		//BISECT CELL ROIs ALONG THE MINOR AXIS OF FITTED ELLIPSES
		phi=Angle[j]*PI/180;
		theta=phi+(PI/2);
		xbl1=(Xc[j]/widthPxl)+(Major[j]/2/widthPxl)*cos(theta); ybl1=(Yc[j]/heightPxl)-(Major[j]/2/heightPxl)*sin(theta); //draw minor axis but with length of major axis (from center to one edge)
		xbl2=(Xc[j]/widthPxl)-(Major[j]/2/widthPxl)*cos(theta); ybl2=(Yc[j]/heightPxl)+(Major[j]/2/heightPxl)*sin(theta); //draw minor axis but with length of major axis (from center to the other edge)
		selectWindow("mask-image_"+i); roiManager("Select", j); setForegroundColor(0, 0, 0); setLineWidth(1); run("Draw", "slice");
		selectWindow("mask-image_"+i); setForegroundColor(0, 0, 0); setLineWidth(1); drawLine(xbl1, ybl1, xbl2, ybl2);
		//FIND COORDINATES OF THE CENTERS OF BOTH HALVES OF THE ELLIPSE
		xh1=(Xc[j]/widthPxl)+(Major[j]/8/widthPxl)*cos(phi);	yh1=(Yc[j]/heightPxl)-(Major[j]/8/heightPxl)*sin(phi); //point on one half of ellipse
		xh2=(Xc[j]/widthPxl)-(Major[j]/8/widthPxl)*cos(phi);	yh2=(Yc[j]/heightPxl)+(Major[j]/8/heightPxl)*sin(phi); //point on the other half
		//CALCULATE DISTANCE OF EACH HALF-ELLIPSE POINT FROM THE NEXT TRAJECTORY POINT CENTROID
		if (j>0)
		{
			dist1=sqrt( pow((yh1-Yc[j-1]),2) + pow((xh1-Xc[j-1]),2) );
			dist2=sqrt( pow((yh2-Yc[j-1]),2) + pow((xh2-Xc[j-1]),2) );
			if (dist2<dist1) //closest half is the Back half, farthest is the Front half
			{
				selectWindow("mask-image_"+i); run("Select None");
				selectWindow("mask-image_"+i); doWand(xh1, yh1); roiManager("add");
				selectWindow("A_"+i); roiManager("Select", nROI); run("Measure");
				FintA[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				selectWindow("B_"+i); roiManager("Select", nROI); run("Measure");
				FintB[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				roiManager("Select", nROI); roiManager("delete");
				selectWindow("mask-image_"+i); run("Select None");
				selectWindow("mask-image_"+i); doWand(xh2, yh2); roiManager("add");
				selectWindow("A_"+i); roiManager("Select", nROI); run("Measure");
				BintA[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				selectWindow("B_"+i); roiManager("Select", nROI); run("Measure");
				BintB[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				roiManager("Select", nROI); roiManager("delete");
			}
			else
			{
				selectWindow("mask-image_"+i); run("Select None");
				selectWindow("mask-image_"+i); doWand(xh2, yh2); roiManager("add");
				selectWindow("A_"+i); roiManager("Select", nROI); run("Measure");
				FintA[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				selectWindow("B_"+i); roiManager("Select", nROI); run("Measure");
				FintB[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				roiManager("Select", nROI); roiManager("delete");
				selectWindow("mask-image_"+i); run("Select None");
				selectWindow("mask-image_"+i); doWand(xh1, yh1); roiManager("add");
				selectWindow("A_"+i); roiManager("Select", nROI); run("Measure");
				BintA[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				selectWindow("B_"+i); roiManager("Select", nROI); run("Measure");
				BintB[j]=parseFloat(getResultString("RawIntDen",nROI));
				Table.deleteRows(nROI, nROI);
				roiManager("Select", nROI); roiManager("delete");
			}
		}
	}
	
	//FROM MEASURED VALUES, CALCULATE ANGLES
	Displacement_Magnitude=newArray(nROI); Displacement_Angle=newArray(nROI);
	Polarization_IndexA=newArray(nROI); Polarization_AngleA=newArray(nROI);	Relative_AngleA=newArray(nROI); //for CHANNEL A
	Polarization_IndexB=newArray(nROI); Polarization_AngleB=newArray(nROI);	Relative_AngleB=newArray(nROI); //for CHANNEL B
	for (j=0; j<=(nROI-1); j++)
	{
		Polarization_IndexA[j]=sqrt( pow((YcMA[j]-Yc[j]),2) + pow((XcMA[j]-Xc[j]),2) );
		Polarization_IndexB[j]=sqrt( pow((YcMB[j]-Yc[j]),2) + pow((XcMB[j]-Xc[j]),2) );
		//atan2 is the correct function, not atan
		t=atan2( (YcMA[j]-Yc[j]) , (XcMA[j]-Xc[j]) );
		if (t<0) { t=abs(t); }
		else { t=2*PI-t; }
		Polarization_AngleA[j]=t*180/PI;
		t=atan2( (YcMB[j]-Yc[j]) , (XcMB[j]-Xc[j]) );
		if (t<0) { t=abs(t); }
		else { t=2*PI-t; }
		Polarization_AngleB[j]=t*180/PI;
		if (j==0) { Displacement_Magnitude[j]=0; Displacement_Angle[j]=0; }
		if (j>0)
		{
			Displacement_Magnitude[j]=sqrt( pow((Yc[j]-Yc[j-1]),2) + pow((Xc[j]-Xc[j-1]),2) );
			t=atan2( (Yc[j]-Yc[j-1]) , (Xc[j]-Xc[j-1]) );
			if (t<0) { t=abs(t); }
			else { t=2*PI-t; }
			Displacement_Angle[j]=t*180/PI;
		}
		Relative_AngleA[j]=Polarization_AngleA[j]-Displacement_Angle[j];
		Relative_AngleB[j]=Polarization_AngleB[j]-Displacement_Angle[j];
	}
	
	if (chkbx2==true)
	{
	//DRAW ARROWS ON THE MASK IMAGE DENOTING THE Polarization_Angle FOR ALL THE CELLS
	selectWindow("mask-image_"+i); roiManager("Show None");
	selectWindow("mask-image_"+i); run("Z Project...", "projection=[Max Intensity]");
	selectWindow("MAX_mask-image_"+i); rename("Add_"+i);
	selectWindow("Add_"+i); run("RGB Color"); run("Multiply...", "value=0"); run("Add...", "value=255"); //to create a white RGB image of same size
	//SCALE UP IMAGE FOR BETTER VISUALIZATION
	scaleUP=10;
	selectWindow("Add_"+i);  wd=getWidth(); ht=getHeight();
	run("Scale...", "x="+scaleUP+" y="+scaleUP+" width="+(scaleUP*wd)+" height="+(scaleUP*ht)+" interpolation=Bilinear average create title=UpScaled");
	selectWindow("Add_"+i); close();
	selectWindow("UpScaled"); run("Duplicate...", "title=[Vectors_Cell_"+i+"] duplicate");
	selectWindow("UpScaled");  close();
	arrowWD=1; arrowSZ=10;
	run("Arrow Tool...", "width="+arrowWD+" size="+arrowSZ+" color=Red style=Notched outline");
	for (j=0; j<nROI; j++)
	{
		if (j>0)
		{
			selectWindow("Vectors_Cell_"+i);			
			setForegroundColor(0, 0, 255);
			makeArrow(round(scaleUP*Xc[j-1]/widthPxl), round(scaleUP*Yc[j-1]/widthPxl), round(scaleUP*Xc[j]/widthPxl), round(scaleUP*Yc[j]/widthPxl), "notched outline");
			Roi.setStrokeWidth(arrowWD); Roi.setStrokeColor("blue"); run("Draw");
		}
		selectWindow("Vectors_Cell_"+i);
		setForegroundColor(0, 255, 0);
		makeArrow(round(scaleUP*Xc[j]/widthPxl), round(scaleUP*Yc[j]/widthPxl), round(scaleUP*XcMA[j]/widthPxl), round(scaleUP*YcMA[j]/widthPxl), "notched outline");
		Roi.setStrokeWidth(arrowWD); Roi.setStrokeColor("green");	run("Draw");
		selectWindow("Vectors_Cell_"+i);
		setForegroundColor(255, 0, 0);
		makeArrow(round(scaleUP*Xc[j]/widthPxl), round(scaleUP*Yc[j]/widthPxl), round(scaleUP*XcMB[j]/widthPxl), round(scaleUP*YcMB[j]/widthPxl), "notched outline");
		Roi.setStrokeWidth(arrowWD); Roi.setStrokeColor("red"); run("Draw");
	}
	setTool("rectangle");
	}
	
	
	//PRINT THE MEASURED VALUES TO LOG FILE
	print("CELL_"+i);
	print("Area_(um^2)"+"\t"+"Circularity"+"\t"+"Displacement_Magnitude_(um)"+"\t"+"Displacement_Angle"+"\t"+"Polarization_Index_A"+"\t"+"Polarization_Angle_A"+"\t"+"Relative_Angle_A"+"\t"+"Polarization_Index_B"+"\t"+"Polarization_Angle_B"+"\t"+"Relative_Angle_B"+"\t"+"FrontA"+"\t"+"FrontB"+"\t"+"BackA"+"\t"+"BackB"+"\t"+"F/B_A"+"\t"+"F/B_B");
	for (j=0; j<=(nROI-1); j++)
	{
		print(Area[j]+"\t"+Circularity[j]+"\t"+Displacement_Magnitude[j]+"\t"+Displacement_Angle[j]+"\t"+Polarization_IndexA[j]+"\t"+Polarization_AngleA[j]+"\t"+Relative_AngleA[j]+"\t"+Polarization_IndexB[j]+"\t"+Polarization_AngleB[j]+"\t"+Relative_AngleB[j]+"\t"+FintA[j]+"\t"+FintB[j]+"\t"+BintA[j]+"\t"+BintB[j]+"\t"+(FintA[j]/BintA[j])+"\t"+(FintB[j]/BintB[j]));
	}
	
	roiManager("save", ROIsaveDestination+"ROIset_Cell-"+i+"_Outline.zip");
	roiManager("Deselect"); roiManager("Delete");
	
	if (chkbx3==false)
	{
		selectWindow("A_"+i); close();
		selectWindow("B_"+i); close();
		selectWindow("AA_"+i); close();
		selectWindow("BB_"+i); close();
	}
	if (chkbx1==false)  { selectWindow("mask-image_"+i); close(); }
	


}
//Cheers! _P.B.